--[[
    LoggingUtil

    This file contains functions to print debug info, warnings and errors.

    DISCLAIMER: For this script to work properly with the printDev functions, the game needs to get started with the
                parameter "-devWarnings"! Otherwise the printDev functions will not print anything!

        @author:    BayernGamers
        @date:      20.12.2025
        @version:   2.2

        History:    v1.0 @30.11.2023 - initial implementation in FS 22
                    -------------------------------------------------------------------------------------------
                    v2.0 @12.11.2024 - convert to FS25
                    -------------------------------------------------------------------------------------------
                    v2.1 @01.04.2025 - added xml DevWarning functions
                    -------------------------------------------------------------------------------------------
                    v2.2 @20.12.2025 - added console command to set debug level at runtime
                    -------------------------------------------------------------------------------------------

        License:    Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]


LoggingUtil = {}
local LoggingUtil_mt = Class(LoggingUtil)

LoggingUtil.DEBUG_LEVELS = {
    LOW = 0,
    MEDIUM = 1,
    HIGH = 2,
    NONE = 3
}

if g_globalMods.g_loggingUtil == nil then
    g_globalMods.g_loggingUtil = {}
    g_globalMods.g_loggingUtil.scriptsByMods = {}
    g_globalMods.g_loggingUtil.consoleCommandAdded = false
end

g_loggingUtil = g_globalMods.g_loggingUtil

-- Create a new LoggingUtil object
-- @param enableDebug: If true, debug messages will be printed
-- @param debugLevel: The debug level of the messages
-- @param luaName: The name of the lua file
-- @return: The Logger
function LoggingUtil.new(enableDebug, debugLevel, luaName)
    local self = setmetatable({}, LoggingUtil_mt)

    self.enableDebug = enableDebug
    self.debugLevel = debugLevel
    self.luaName = luaName

    if g_loggingUtil.scriptsByMods[g_currentModName] == nil then
        g_loggingUtil.scriptsByMods[g_currentModName] = {}
    end

    if g_loggingUtil.scriptsByMods[g_currentModName][luaName] == nil then
        g_loggingUtil.scriptsByMods[g_currentModName][luaName] = self
    end

    if g_loggingUtil.consoleCommandAdded == false then
        g_loggingUtil.consoleCommandAdded = true
        addConsoleCommand("gsLoggingUtilSetDebugLevel", "Set the debug level for a script", "consoleCommandSetDebugLevel", LoggingUtil, "[modName]; [scriptName]; [debugLevel]")
        addConsoleCommand("gsLoggingUtilListScriptsByMod", "List all scripts registered for a mod", "consoleCommandListScriptsByMod", LoggingUtil, "[modName]")
    end

    return self
end

function LoggingUtil.consoleCommandSetDebugLevel(_, modName, scriptName, debugLevelStr)
    if not string.startsWith(modName, "FS25_") then
        modName = "FS25_" .. modName
    end

    if g_loggingUtil.scriptsByMods[modName] == nil then
        Logging.error("Mod not found: " .. tostring(modName))
        return
    end

    if not string.endsWith(scriptName, ".lua") then
        scriptName = scriptName .. ".lua"
    end

    if g_loggingUtil.scriptsByMods[modName][scriptName] == nil then
        Logging.error("Script not found: " .. tostring(scriptName) .. " in mod " .. tostring(modName))
        return
    end

    local debugLevel = LoggingUtil.DEBUG_LEVELS[debugLevelStr:upper()]
    if debugLevel == nil then
        Logging.error("Invalid debug level: " .. tostring(debugLevelStr) .. ". Valid levels are [LOW | MEDIUM | HIGH | NONE]")
        return
    end

    local logger = g_loggingUtil.scriptsByMods[modName][scriptName]
    logger.debugLevel = debugLevel
    Logging.info("Set debug level for '" .. tostring(scriptName) .. "' in mod '" .. tostring(modName) .. "' to '" .. tostring(debugLevelStr) .. "'")
end

function LoggingUtil.consoleCommandListScriptsByMod(_, modName)
    if not string.startsWith(modName, "FS25_") then
        modName = "FS25_" .. modName
    end

    if g_loggingUtil.scriptsByMods[modName] == nil then
        Logging.error("Mod not found: " .. tostring(modName))
        return
    end

    print("Scripts registered for mod '" .. tostring(modName) .. "':")
    for scriptName, logger in pairs(g_loggingUtil.scriptsByMods[modName]) do
        local debugLevelStr = "UNKNOWN"
        for levelStr, levelValue in pairs(LoggingUtil.DEBUG_LEVELS) do
            if levelValue == logger.debugLevel then
                debugLevelStr = levelStr
                break
            end
        end
        print(" - " .. tostring(scriptName) .. " (Debug Level: " .. tostring(debugLevelStr) .. ")")
    end
end

-- Print debug info
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevInfo(message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.devInfo(prefix .. message)
    end
end

-- Print debug warning
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevWarning(message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.devWarning(prefix .. message)
    end
end

-- Print debug error
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevError(message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.devError(prefix .. message)
    end
end

-- Print info
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printInfo(message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.info(prefix .. message)
end

-- Print warning
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printWarning(message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.warning(prefix .. message)
end

-- Print error
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printError(message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.error(prefix .. message)
end

-- Print XML info
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printXMLInfo(xmlFile, message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.xmlInfo(xmlFile, prefix .. message)
end

-- Print XML warning
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printXMLWarning(xmlFile, message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.xmlWarning(xmlFile, prefix .. message)
end

-- Print XML error
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printXMLError(xmlFile, message)
    local prefix = ""
    prefix = "::" .. self.luaName .. ":: "
    Logging.xmlError(xmlFile, prefix .. message)
end

-- Print XML info with debug level
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevXMLInfo(xmlFile, message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.xmlDevInfo(xmlFile, prefix .. message)
    end
end

-- Print XML warning with debug level
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevXMLWarning(xmlFile, message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.xmlDevWarning(xmlFile, prefix .. message)
    end
end

-- Print XML error with debug level
-- @param xmlFile: the xml file
-- @param message: the message to print
-- @param debugLevel: the debug level of the message
-- @param addPrefix: if true, the luaName will be added as prefix to the message
-- @param luaName: the name of the lua file
function LoggingUtil:printDevXMLError(xmlFile, message, debugLevel)
    if self.enableDebug and debugLevel >= self.debugLevel then
        local prefix = ""
        prefix = "::" .. self.luaName .. ":: "
        Logging.xmlDevError(xmlFile, prefix .. message)
    end
end
